---------------------------------------------------------------
--
--  RAPID - Rapid Ada Portable Interface Designer
--
--  MENU_IO.ADB
--  Description : Read/Write Menu to file
--
--  Copyright (C) 1999, Martin C. Carlisle <carlislem@acm.org>
--
-- RAPID is free software; you can redistribute it and/or
-- modify it without restriction.  However, we ask that you
-- please retain the original author information, and clearly
-- indicate if it has been modified.
--
-- RAPID is distributed in the hope that it will be useful,
-- but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
--
-- As a special exception, if other files instantiate generics from
-- this unit, or you link this unit with other files to produce an
-- executable, this unit does not by itself cause the resulting
-- executable to be covered by the GNU General Public License.
-- This exception does not however invalidate any other reasons
-- why the executable file might be covered by the GNU Public
-- License.
---------------------------------------------------------------
-- Change log:
-- 09/20/99 (mcc) : added Separator
---------------------------------------------------------------

with Ada.Text_IO;
with Gui_Enum;
with File_Helpers;
with Gui.Menu;
with Menu_Edit;

package body Menu_IO is

   use type Gui_Enum.Keyword;
   use type Gui.Menu.Menu_Pointer, Gui.String_Pointer;

   subtype Word_String is String (1 .. 80);

   -- This is used so we can generate unique separators even
   -- if the menu is never displayed
   Separator_Number : Natural := 1;

   -- assumes the keyword "menubar" has already been read, and
   -- reads the rest of the menu from the file
   --
   -- This is an LL(1) grammar:
   -- <menubar> -> MENUBAR <menulist> MENUBAR
   -- <menulist> -> MENU <submenuinfo> <menulist> |
   --               ITEM <iteminfo> <menulist> | END
   -- <submenuinfo> -> <name> <underline> <possible_action>
   -- <possible_action> -> <action> | null
   -- <iteminfo> -> <name> <underline> <action> <accelerator>
   -- <accelerator> -> <accel_key> | null
   --
   -- an example of the menubar section of the file:
   -- MENUBAR
   --   MENU File 1 Menu_Action
   --     ITEM New  1 New_Action Ctrl+N
   --     ITEM Open 1 Open_Action Ctrl+O
   --     MENU Submenu 1
   --       ITEM "Sub Item 1" Sub_Action
   --     ENDOF MENU
   --   ENDOF MENU
   --   MENU Edit 1
   --     ITEM Cut 3 Cut_Action
   --     ITEM Copy 1 Copy_Action
   --   ENDOF MENU
   -- ENDOF MENUBAR

   procedure Read_Menubar
     (Menubar : out gui.Menu.Menu_Pointer;
      Count   : in out Integer)
   is

      function Read_Item return gui.Menu.Menu_Access is
         Result        : gui.Menu.Menu_Item_Pointer := new gui.Menu.Menu_Item;
         Name_String   : Word_String;
         Name_Length   : Natural;
         Underline     : Natural;
         Action_String : Word_String;
         Action_Length : Natural;
         Accel_String  : Word_String;
         Accel_Length  : Natural;
      begin
         -- read information
         File_Helpers.Get_String
           (Token_Index => 2,
            Item        => Name_String,
            Last        => Name_Length);
         Underline := Natural'Value (File_Helpers.Token (3).all);
         File_Helpers.Get_String
           (Token_Index => 4,
            Item        => Action_String,
            Last        => Action_Length);
         File_Helpers.Get_String
           (Token_Index => 5,
            Item        => Accel_String,
            Last        => Accel_Length);

         -- store in record
         --Result.Number := Count;
         Count := Count + 1;

         Result.Name               :=
           new String'(Name_String (1 .. Name_Length));
         Result.Underline_Position := Underline;
         Result.Action             :=
           new String'(Action_String (1 .. Action_Length));
         if Accel_Length > 0 then
            Result.Accelerator :=
              new String'(Accel_String (1 .. Accel_Length));
         else
            Result.Accelerator := new String'("");
         end if;

         return gui.Menu.Menu_Access (Result);
      end Read_Item;

      function Read_Menu return gui.Menu.Menu_Access is
         Result        : gui.Menu.Submenu_Pointer := new gui.Menu.Submenu;
         Name_String   : Word_String;
         Name_Length   : Natural;
         Underline     : Natural;
         Action_String : Word_String;
         Action_Length : Natural;
      begin
         -- read information
         File_Helpers.Get_String
           (Token_Index => 2,
            Item        => Name_String,
            Last        => Name_Length);
         Underline := Natural'Value (File_Helpers.Token (3).all);
         File_Helpers.Get_String
           (Token_Index => 4,
            Item        => Action_String,
            Last        => Action_Length);

         -- store in record
         --Result.Number := Count;
         Count := Count + 1;

         Result.Name               :=
           new String'(Name_String (1 .. Name_Length));
         Result.Underline_Position := Underline;
         if Action_Length > 0 then
            Result.Action :=
              new String'(Action_String (1 .. Action_Length));
         else
            Result.Action := null;
         end if;

         Read_Menubar (Result.Items, Count);

         return gui.Menu.Menu_Access (Result);
      end Read_Menu;

      Last    : gui.Menu.Menu_Access;
      Keyword : Gui_Enum.Keyword;
   begin -- Read_Menubar
      gui.Menu.Menu_List_Package.Initialize (Menubar);
      loop
         File_Helpers.Get_Line;
         Keyword := Gui_Enum.Keyword'Value (File_Helpers.Token (1).all);

         case Keyword is
            when Gui_Enum.EndOf =>
               Keyword := Gui_Enum.Keyword'Value (File_Helpers.Token (2).all);
               if Keyword = Gui_Enum.Menu
                 or else Keyword = Gui_Enum.Menubar
               then
                  exit;
               else
                  raise Bad_File;
               end if;
            when Gui_Enum.Item =>
               Last := Read_Item;
            when Gui_Enum.Menu =>
               Last := Read_Menu;
            when Gui_Enum.Separator =>
               Last      := new gui.Menu.Separator;
               Last.Name := Menu_Edit.Separator_Name'Access;
               -- note that separators need to be numbered in
               -- case we generate w/o displaying
               Last.Number      := Separator_Number;
               Separator_Number := Separator_Number + 1;
            when others =>
               raise Bad_File;
         end case;

         gui.Menu.Menu_List_Package.AddToRear (Menubar, Last);
      end loop;

   exception
      when others =>
         raise Bad_File;
   end Read_Menubar;

   -- writes whole section described above
   procedure Write_Menubar (Menubar : gui.Menu.Menu_Pointer) is
      procedure Write_Submenu (Submenu : gui.Menu.Submenu);

      procedure Write_Menu_Item (Menu_Item : gui.Menu.Menu_Item)
      is
      begin
         File_Helpers.Put (Gui_Enum.Img (Gui_Enum.Item) & " ");
         File_Helpers.Put_String (Menu_Item.Name.all);
         File_Helpers.Put ("""");
         File_Helpers.Put (Natural'Image (Menu_Item.Underline_Position));
         File_Helpers.Put (" """);
         File_Helpers.Put_String (Menu_Item.Action.all);
         File_Helpers.Put ("""");
         if Menu_Item.Accelerator /= null then
            File_Helpers.Put (" " & Menu_Item.Accelerator.all);
         end if;
         File_Helpers.P;
      end Write_Menu_Item;

      procedure Write_Menulist (Menulist : gui.Menu.Menu_Pointer)
      is
         Current_Position : gui.Menu.Menu_Position :=
            gui.Menu.Menu_List_Package.First (Menulist);
         Current_Menu     : gui.Menu.Menu_Access;
      begin
         while not gui.Menu.Menu_List_Package.IsPastEnd
                     (Menulist,
                      Current_Position)
         loop
            Current_Menu :=
               gui.Menu.Menu_List_Package.Retrieve
                 (Menulist,
                  Current_Position);
            if Current_Menu.all in gui.Menu.Submenu'Class then
               Write_Submenu (gui.Menu.Submenu (Current_Menu.all));
            elsif Current_Menu.all in gui.Menu.Menu_Item'Class then
               Write_Menu_Item (gui.Menu.Menu_Item (Current_Menu.all));
            elsif Current_Menu.all in gui.Menu.Separator'Class then
               File_Helpers.P (Gui_Enum.Img (Gui_Enum.Separator), Indent => False);
            else
               raise Constraint_Error;
            end if;
            gui.Menu.Menu_List_Package.GoAhead (Menulist, Current_Position);
         end loop;
      end Write_Menulist;

      procedure Write_Submenu (Submenu : gui.Menu.Submenu) is
      begin
         File_Helpers.Put (Gui_Enum.Img (Gui_Enum.Menu) & " """);
         File_Helpers.Put_String (Submenu.Name.all);
         File_Helpers.Put ("""");
         File_Helpers.Put (Natural'Image (Submenu.Underline_Position));
         if Submenu.Action /= null then
            File_Helpers.Put (" """);
            File_Helpers.Put_String (Submenu.Action.all);
            File_Helpers.Put ("""");
         end if;
         File_Helpers.P;

         Write_Menulist (Submenu.Items);

         File_Helpers.Put (Gui_Enum.Img (Gui_Enum.EndOf) & " ");
         File_Helpers.Put (Gui_Enum.Img (Gui_Enum.Menu));
         File_Helpers.P;
      end Write_Submenu;

   begin -- Write_Menubar
      File_Helpers.P (Gui_Enum.Img (Gui_Enum.Menubar), Indent => False);

      Write_Menulist (Menubar);

      File_Helpers.Put (Gui_Enum.Img (Gui_Enum.EndOf) & " ");
      File_Helpers.Put (Gui_Enum.Img (Gui_Enum.Menubar));
      File_Helpers.P;
   end Write_Menubar;

end Menu_IO;

